<?php
/**
 * API Endpoint for Recent Activity
 * Returns recent activity data for notifications
 */

require_once '../includes/config.php';

header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    $action = $_GET['action'] ?? 'recent';
    
    switch ($action) {
        case 'recent':
            // Get recent activity (check-ins and SOS)
            $stmt = $pdo->query("
                SELECT 
                    'checkin' as type,
                    c.id,
                    u.name as user_name,
                    c.location_name,
                    c.created_at,
                    c.status,
                    'camera' as icon,
                    CONCAT('Checked in at ', COALESCE(c.location_name, 'Unknown location')) as title,
                    'verified' as status_type
                FROM check_ins c
                JOIN users u ON c.user_id = u.id
                UNION ALL
                SELECT 
                    'sos' as type,
                    s.id,
                    u.name as user_name,
                    s.location_name,
                    s.created_at,
                    s.status,
                    'exclamation-triangle' as icon,
                    CONCAT('SOS Alert - ', u.name) as title,
                    s.status as status_type
                FROM sos_alerts s
                JOIN users u ON s.user_id = u.id
                ORDER BY created_at DESC
                LIMIT 20
            ");
            $activities = $stmt->fetchAll();
            
            $data = array_map(function($activity) {
                return [
                    'id' => $activity['id'],
                    'type' => $activity['type'],
                    'title' => $activity['title'],
                    'description' => $activity['location_name'] ?? '',
                    'time' => timeAgo($activity['created_at']),
                    'icon' => $activity['icon'],
                    'status' => $activity['status_type']
                ];
            }, $activities);
            
            echo json_encode(['success' => true, 'data' => $data]);
            break;
            
        case 'stats':
            // Get quick stats
            $stats = [
                'total_staff' => $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn(),
                'on_duty' => $pdo->query("SELECT COUNT(*) FROM users WHERE is_on_duty = 1 AND status = 'active'")->fetchColumn(),
                'today_checkins' => $pdo->query("SELECT COUNT(*) FROM check_ins WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
                'active_sos' => $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn()
            ];
            echo json_encode(['success' => true, 'data' => $stats]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}
