<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'Live Tracking Map';
$admin = getCurrentAdmin();

// Get all staff with their latest locations
$pdo = getDBConnection();
$staffLocations = $pdo->query("
    SELECT 
        u.id, 
        u.name, 
        u.email,
        u.phone,
        u.is_on_duty,
        u.status as user_status,
        l.latitude,
        l.longitude,
        l.location_name,
        l.created_at as last_location_update,
        (SELECT COUNT(*) FROM sos_alerts s WHERE s.user_id = u.id AND s.status = 'active') as active_sos
    FROM users u
    LEFT JOIN (
        SELECT user_id, latitude, longitude, location_name, created_at
        FROM location_logs
        WHERE id IN (
            SELECT MAX(id) FROM location_logs GROUP BY user_id
        )
    ) l ON l.user_id = u.id
    WHERE u.status = 'active'
    ORDER BY u.name
")->fetchAll();

// Get active SOS count for header
$sosCount = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

include 'includes/header.php';
?>

<!-- Map Section -->
<div class="bg-white rounded-xl p-6 shadow-sm mb-6 fade-in">
    <div class="flex items-center justify-between mb-4">
        <h3 class="text-lg font-semibold text-gray-800">Live Staff Locations</h3>
        <div class="flex gap-2">
            <button onclick="filterMarkers('all')" class="px-3 py-1 bg-gray-100 text-gray-700 rounded-full text-sm hover:bg-gray-200">
                <i class="fas fa-circle mr-1 text-green-500"></i>All
            </button>
            <button onclick="filterMarkers('safe')" class="px-3 py-1 bg-green-100 text-green-700 rounded-full text-sm hover:bg-green-200">
                <i class="fas fa-circle mr-1"></i>Safe
            </button>
            <button onclick="filterMarkers('sos')" class="px-3 py-1 bg-red-100 text-red-700 rounded-full text-sm hover:bg-red-200">
                <i class="fas fa-circle mr-1"></i>SOS
            </button>
            <button onclick="filterMarkers('offline')" class="px-3 py-1 bg-gray-100 text-gray-700 rounded-full text-sm hover:bg-gray-200">
                <i class="fas fa-circle mr-1"></i>Offline
            </button>
            <button onclick="refreshLocations()" class="px-3 py-1 bg-primary text-white rounded-full text-sm hover:bg-primary-dark">
                <i class="fas fa-sync-alt mr-1"></i>Refresh
            </button>
        </div>
    </div>
    
    <!-- Map Container -->
    <div id="map" class="w-full h-96 bg-gray-100 rounded-lg flex items-center justify-center relative">
        <div id="map-loading" class="absolute inset-0 bg-white/80 flex items-center justify-center z-10">
            <div class="text-center">
                <i class="fas fa-spinner fa-spin text-3xl text-primary mb-2"></i>
                <p class="text-gray-500">Loading map...</p>
            </div>
        </div>
        <p class="text-gray-500" id="map-placeholder">
            <i class="fas fa-map-marked-alt text-4xl mb-2 block"></i>
            Google Maps will be displayed here with staff markers
        </p>
    </div>
</div>

<!-- Staff List with Live Status -->
<div class="bg-white rounded-xl p-6 shadow-sm fade-in">
    <h3 class="text-lg font-semibold text-gray-800 mb-4">Staff Status</h3>
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        <?php foreach ($staffLocations as $staff): ?>
        <div class="border rounded-lg p-4 staff-card" data-status="<?php echo $staff['active_sos'] > 0 ? 'sos' : ($staff['is_on_duty'] ? 'safe' : 'offline'); ?>">
            <div class="flex items-start justify-between">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-full bg-primary text-white flex items-center justify-center font-semibold">
                        <?php echo strtoupper(substr($staff['name'], 0, 2)); ?>
                    </div>
                    <div>
                        <p class="font-medium text-gray-800"><?php echo htmlspecialchars($staff['name']); ?></p>
                        <p class="text-sm text-gray-500"><?php echo htmlspecialchars($staff['phone'] ?? 'No phone'); ?></p>
                    </div>
                </div>
                <div>
                    <?php if ($staff['active_sos'] > 0): ?>
                    <span class="px-2 py-1 bg-red-100 text-red-700 rounded-full text-xs font-medium">
                        <i class="fas fa-exclamation-triangle mr-1"></i>SOS
                    </span>
                    <?php elseif ($staff['is_on_duty']): ?>
                    <span class="px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs font-medium">
                        <i class="fas fa-circle mr-1 text-green-500" style="font-size: 8px;"></i>On Duty
                    </span>
                    <?php else: ?>
                    <span class="px-2 py-1 bg-gray-100 text-gray-600 rounded-full text-xs font-medium">
                        <i class="fas fa-circle mr-1 text-gray-400" style="font-size: 8px;"></i>Off Duty
                    </span>
                    <?php endif; ?>
                </div>
            </div>
            <?php if ($staff['latitude']): ?>
            <div class="mt-3 pt-3 border-t text-sm">
                <p class="text-gray-500">
                    <i class="fas fa-map-marker-alt mr-1"></i>
                    <?php echo htmlspecialchars($staff['location_name'] ?? 'Unknown location'); ?>
                </p>
                <p class="text-gray-400 text-xs mt-1">
                    Last update: <?php echo timeAgo($staff['last_location_update']); ?>
                </p>
            </div>
            <?php else: ?>
            <div class="mt-3 pt-3 border-t text-sm">
                <p class="text-gray-400">
                    <i class="fas fa-map-slash mr-1"></i>
                    No location data
                </p>
            </div>
            <?php endif; ?>
            <div class="mt-3 flex gap-2">
                <?php if ($staff['latitude']): ?>
                <button onclick="focusOnMap(<?php echo $staff['latitude']; ?>, <?php echo $staff['longitude']; ?>, '<?php echo htmlspecialchars($staff['name']); ?>')" 
                    class="flex-1 px-3 py-2 bg-primary/10 text-primary rounded-lg text-sm hover:bg-primary/20">
                    <i class="fas fa-crosshairs mr-1"></i>Locate
                </button>
                <?php endif; ?>
                <a href="staff.php?view=<?php echo $staff['id']; ?>" class="flex-1 px-3 py-2 bg-gray-100 text-gray-700 rounded-lg text-sm hover:bg-gray-200 text-center">
                    <i class="fas fa-user mr-1"></i>Profile
                </a>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Hidden data for JavaScript -->
<script>
    const staffLocations = <?php echo json_encode(array_map(function($s) {
        return [
            'id' => $s['id'],
            'name' => $s['name'],
            'lat' => floatval($s['latitude'] ?? 0),
            'lng' => floatval($s['longitude'] ?? 0),
            'location' => $s['location_name'] ?? '',
            'status' => $s['active_sos'] > 0 ? 'sos' : ($s['is_on_duty'] ? 'safe' : 'offline'),
            'lastUpdate' => $s['last_location_update'] ?? null
        ];
    }, $staffLocations)); ?>;
    
    let map;
    let markers = [];
    
    function initMap() {
        const defaultCenter = { lat: 6.5244, lng: 3.3792 }; // Lagos coordinates
        
        try {
            map = new google.maps.Map(document.getElementById('map'), {
                zoom: 14,
                center: defaultCenter,
                styles: [
                    {
                        "featureType": "poi",
                        "elementType": "labels",
                        "stylers": [{ "visibility": "off" }]
                    }
                ]
            });
            
            // Add markers for all staff
            staffLocations.forEach(addMarker);
            
            // Hide loading
            document.getElementById('map-loading').style.display = 'none';
            document.getElementById('map-placeholder').style.display = 'none';
        } catch (e) {
            console.log('Google Maps not loaded:', e);
            document.getElementById('map-loading').innerHTML = `
                <div class="text-center">
                    <i class="fas fa-exclamation-triangle text-3xl text-yellow-500 mb-2"></i>
                    <p class="text-gray-500">Configure Google Maps API key to enable live map</p>
                    <p class="text-sm text-gray-400 mt-1">Edit map.php and add your API key</p>
                </div>
            `;
        }
    }
    
    function addMarker(staff) {
        if (!staff.lat || !staff.lng) return;
        
        const iconColor = staff.status === 'sos' ? '#ef4444' : 
                         staff.status === 'safe' ? '#22c55e' : '#94a3b8';
        
        const marker = new google.maps.Marker({
            position: { lat: staff.lat, lng: staff.lng },
            map: map,
            title: staff.name,
            icon: {
                path: google.maps.SymbolPath.CIRCLE,
                scale: 10,
                fillColor: iconColor,
                fillOpacity: 1,
                strokeColor: '#ffffff',
                strokeWeight: 2
            }
        });
        
        const infoWindow = new google.maps.InfoWindow({
            content: `
                <div class="p-2">
                    <p class="font-semibold">${staff.name}</p>
                    <p class="text-sm text-gray-600">${staff.location || 'Unknown location'}</p>
                    <p class="text-xs text-gray-400">Status: ${staff.status}</p>
                </div>
            `
        });
        
        marker.addListener('click', () => {
            infoWindow.open(map, marker);
        });
        
        markers.push({ marker, staff });
    }
    
    function filterMarkers(status) {
        markers.forEach(({ marker, staff }) => {
            if (status === 'all' || staff.status === status) {
                marker.setVisible(true);
            } else {
                marker.setVisible(false);
            }
        });
    }
    
    function focusOnMap(lat, lng, name) {
        if (map) {
            map.setCenter({ lat, lng });
            map.setZoom(17);
            
            // Find and open the marker info
            const markerData = markers.find(m => m.staff.name === name);
            if (markerData) {
                // Could trigger info window here
            }
        }
    }
    
    function refreshLocations() {
        location.reload();
    }
    
    // Load map when page loads
    document.addEventListener('DOMContentLoaded', initMap);
</script>

<?php
function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
