<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'SOS Alerts';
$admin = getCurrentAdmin();

$pdo = getDBConnection();

// Get active SOS count for header
$sosCount = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

// Handle form submissions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'acknowledge':
                    $sosId = (int)$_POST['sos_id'];
                    
                    $stmt = $pdo->prepare("UPDATE sos_alerts SET status = 'acknowledged' WHERE id = ?");
                    $stmt->execute([$sosId]);
                    
                    $message = 'SOS alert acknowledged!';
                    $messageType = 'success';
                    logAdminAction('acknowledge_sos', "Acknowledged SOS ID: $sosId");
                    break;
                    
                case 'resolve':
                    $sosId = (int)$_POST['sos_id'];
                    $notes = sanitize($_POST['notes'] ?? '');
                    
                    $stmt = $pdo->prepare("UPDATE sos_alerts SET status = 'resolved', resolved_by = ?, resolved_at = NOW(), description = CONCAT(IFNULL(description, ''), ?)");
                    $stmt->execute([$admin['id'], "\n[Resolved] " . $notes]);
                    
                    $message = 'SOS alert resolved!';
                    $messageType = 'success';
                    logAdminAction('resolve_sos', "Resolved SOS ID: $sosId");
                    break;
                    
                case 'false_alarm':
                    $sosId = (int)$_POST['sos_id'];
                    $notes = sanitize($_POST['notes'] ?? '');
                    
                    $stmt = $pdo->prepare("UPDATE sos_alerts SET status = 'false_alarm', resolved_by = ?, resolved_at = NOW(), description = CONCAT(IFNULL(description, ''), ?)");
                    $stmt->execute([$admin['id'], "\n[False Alarm] " . $notes]);
                    
                    $message = 'Marked as false alarm!';
                    $messageType = 'success';
                    logAdminAction('false_alarm_sos', "Marked false alarm SOS ID: $sosId");
                    break;
                    
                case 'delete':
                    $sosId = (int)$_POST['sos_id'];
                    
                    $stmt = $pdo->prepare("DELETE FROM sos_alerts WHERE id = ?");
                    $stmt->execute([$sosId]);
                    
                    $message = 'SOS alert deleted!';
                    $messageType = 'success';
                    break;
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $messageType = 'error';
        }
    }
}

// Get filter parameters
$statusFilter = sanitize($_GET['status'] ?? '');

// Build query
$query = "
    SELECT s.*, u.name as user_name, u.phone as user_phone,
           a.name as resolved_by_name
    FROM sos_alerts s
    JOIN users u ON s.user_id = u.id
    LEFT JOIN admins a ON s.resolved_by = a.id
    WHERE 1=1
";

$params = [];

if ($statusFilter) {
    $query .= " AND s.status = ?";
    $params[] = $statusFilter;
} else {
    // Default to showing active first
    $query .= " AND s.status = 'active'";
}

$query .= " ORDER BY 
    CASE s.status 
        WHEN 'active' THEN 1 
        WHEN 'acknowledged' THEN 2 
        ELSE 3 
    END, 
    s.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$alerts = $stmt->fetchAll();

// Get statistics
$stats = [
    'active' => $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn(),
    'acknowledged' => $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'acknowledged'")->fetchColumn(),
    'resolved' => $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'resolved'")->fetchColumn(),
    'today' => $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE DATE(created_at) = CURDATE()")->fetchColumn()
];

include 'includes/header.php';
?>

<!-- SOS Alerts -->
<div class="fade-in">
    <?php if ($message): ?>
    <div class="mb-6 p-4 rounded-lg flex items-center gap-2 <?php echo $messageType === 'success' ? 'bg-green-50 text-green-700 border border-green-200' : 'bg-red-50 text-red-700 border border-red-200'; ?>">
        <i class="fas fa-<?php echo $messageType === 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i>
        <?php echo htmlspecialchars($message); ?>
    </div>
    <?php endif; ?>

    <!-- Stats -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-red-50 rounded-xl p-4 shadow-sm border border-red-100">
            <p class="text-red-600 text-sm">Active Alerts</p>
            <p class="text-3xl font-bold text-red-600"><?php echo $stats['active']; ?></p>
        </div>
        <div class="bg-yellow-50 rounded-xl p-4 shadow-sm border border-yellow-100">
            <p class="text-yellow-700 text-sm">Acknowledged</p>
            <p class="text-2xl font-bold text-yellow-600"><?php echo $stats['acknowledged']; ?></p>
        </div>
        <div class="bg-green-50 rounded-xl p-4 shadow-sm border border-green-100">
            <p class="text-green-700 text-sm">Resolved</p>
            <p class="text-2xl font-bold text-green-600"><?php echo $stats['resolved']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Today's Alerts</p>
            <p class="text-2xl font-bold text-gray-800"><?php echo $stats['today']; ?></p>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-xl p-4 shadow-sm mb-6">
        <form method="GET" class="flex flex-wrap items-center gap-4">
            <div class="flex items-center gap-2">
                <label class="text-sm text-gray-600">Status:</label>
                <select name="status" class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
                    <option value="">All</option>
                    <option value="active" <?php echo $statusFilter === 'active' ? 'selected' : ''; ?>>Active</option>
                    <option value="acknowledged" <?php echo $statusFilter === 'acknowledged' ? 'selected' : ''; ?>>Acknowledged</option>
                    <option value="resolved" <?php echo $statusFilter === 'resolved' ? 'selected' : ''; ?>>Resolved</option>
                    <option value="false_alarm" <?php echo $statusFilter === 'false_alarm' ? 'selected' : ''; ?>>False Alarm</option>
                </select>
            </div>
            
            <button type="submit" class="px-4 py-2 bg-primary text-white rounded-lg text-sm hover:bg-primary-dark">
                <i class="fas fa-filter mr-1"></i> Filter
            </button>
            
            <?php if ($statusFilter): ?>
            <a href="sos.php" class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg text-sm hover:bg-gray-200">
                <i class="fas fa-times mr-1"></i> Clear
            </a>
            <?php endif; ?>
        </form>
    </div>

    <!-- Alerts List -->
    <div class="space-y-4">
        <?php if (count($alerts) > 0): ?>
            <?php foreach ($alerts as $alert): ?>
            <div class="bg-white rounded-xl shadow-sm overflow-hidden <?php echo $alert['status'] === 'active' ? 'border-l-4 border-red-500' : ($alert['status'] === 'acknowledged' ? 'border-l-4 border-yellow-500' : 'border-l-4 border-green-500'); ?>">
                <div class="p-6">
                    <div class="flex items-start justify-between">
                        <div class="flex items-start gap-4">
                            <!-- Status Icon -->
                            <div class="w-12 h-12 rounded-full flex items-center justify-center <?php echo $alert['status'] === 'active' ? 'bg-red-100' : ($alert['status'] === 'acknowledged' ? 'bg-yellow-100' : 'bg-green-100'); ?>">
                                <i class="fas fa-exclamation-triangle text-xl <?php echo $alert['status'] === 'active' ? 'text-red-600' : ($alert['status'] === 'acknowledged' ? 'text-yellow-600' : 'text-green-600'); ?>"></i>
                            </div>
                            
                            <!-- Alert Info -->
                            <div>
                                <div class="flex items-center gap-2 mb-1">
                                    <h3 class="text-lg font-semibold text-gray-800">SOS Alert - <?php echo htmlspecialchars($alert['user_name']); ?></h3>
                                    <?php if ($alert['status'] === 'active'): ?>
                                    <span class="px-2 py-1 bg-red-100 text-red-700 rounded-full text-xs font-medium animate-pulse">
                                        ACTIVE
                                    </span>
                                    <?php elseif ($alert['status'] === 'acknowledged'): ?>
                                    <span class="px-2 py-1 bg-yellow-100 text-yellow-700 rounded-full text-xs font-medium">
                                        Acknowledged
                                    </span>
                                    <?php else: ?>
                                    <span class="px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs font-medium">
                                        Resolved
                                    </span>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="flex items-center gap-4 text-sm text-gray-600 mb-2">
                                    <span><i class="fas fa-user mr-1"></i> <?php echo htmlspecialchars($alert['user_phone'] ?? 'No phone'); ?></span>
                                    <span><i class="fas fa-clock mr-1"></i> <?php echo timeAgo($alert['created_at']); ?></span>
                                </div>
                                
                                <div class="flex items-center gap-2 text-sm text-gray-600">
                                    <i class="fas fa-map-marker-alt mr-1"></i>
                                    <?php echo htmlspecialchars($alert['location_name'] ?? 'Unknown location'); ?>
                                    <span class="text-gray-400">(<?php echo $alert['latitude']; ?>, <?php echo $alert['longitude']; ?>)</span>
                                </div>
                                
                                <?php if ($alert['description']): ?>
                                <div class="mt-3 p-3 bg-gray-50 rounded-lg">
                                    <p class="text-sm text-gray-700"><?php echo nl2br(htmlspecialchars($alert['description'])); ?></p>
                                </div>
                                <?php endif; ?>
                                
                                <?php if ($alert['resolved_by_name']): ?>
                                <div class="mt-2 text-sm text-gray-500">
                                    <i class="fas fa-check-circle mr-1"></i>
                                    Resolved by <?php echo htmlspecialchars($alert['resolved_by_name']); ?>
                                    <?php if ($alert['resolved_at']): ?>
                                    at <?php echo date('M d, Y H:i', strtotime($alert['resolved_at'])); ?>
                                    <?php endif; ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Actions -->
                        <div class="flex flex-col gap-2">
                            <?php if ($alert['status'] === 'active'): ?>
                            <form method="POST">
                                <input type="hidden" name="action" value="acknowledge">
                                <input type="hidden" name="sos_id" value="<?php echo $alert['id']; ?>">
                                <button type="submit" class="px-4 py-2 bg-yellow-500 text-white rounded-lg text-sm hover:bg-yellow-600">
                                    <i class="fas fa-check mr-1"></i> Acknowledge
                                </button>
                            </form>
                            <?php endif; ?>
                            
                            <?php if (in_array($alert['status'], ['active', 'acknowledged'])): ?>
                            <button onclick="openResolveModal(<?php echo $alert['id']; ?>, '<?php echo htmlspecialchars($alert['user_name']); ?>')" 
                                class="px-4 py-2 bg-green-500 text-white rounded-lg text-sm hover:bg-green-600">
                                <i class="fas fa-shield-alt mr-1"></i> Confirm Safe
                            </button>
                            <?php endif; ?>
                            
                            <?php if ($alert['status'] === 'active'): ?>
                            <button onclick="openFalseAlarmModal(<?php echo $alert['id']; ?>, '<?php echo htmlspecialchars($alert['user_name']); ?>')" 
                                class="px-4 py-2 bg-gray-500 text-white rounded-lg text-sm hover:bg-gray-600">
                                <i class="fas fa-times mr-1"></i> False Alarm
                            </button>
                            <?php endif; ?>
                            
                            <a href="map.php?focus=<?php echo $alert['user_id']; ?>" class="px-4 py-2 bg-blue-500 text-white rounded-lg text-sm hover:bg-blue-600 text-center">
                                <i class="fas fa-map-marker-alt mr-1"></i> View Location
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        <?php else: ?>
        <div class="bg-white rounded-xl p-12 shadow-sm text-center">
            <div class="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <i class="fas fa-check-circle text-4xl text-green-600"></i>
            </div>
            <h3 class="text-xl font-semibold text-gray-800 mb-2">All Clear!</h3>
            <p class="text-gray-500">No active SOS alerts at the moment.</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Resolve Modal -->
<div id="resolveModal" class="fixed inset-0 bg-black/50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl w-full max-w-md mx-4">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">Confirm Safe - <span id="resolveUserName"></span></h3>
            <button onclick="closeModal('resolveModal')" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form method="POST" class="p-4">
            <input type="hidden" name="action" value="resolve">
            <input type="hidden" name="sos_id" id="resolveSosId">
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Resolution Notes (optional)</label>
                <textarea name="notes" rows="3" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary"
                    placeholder="Add notes about the resolution..."></textarea>
            </div>
            
            <div class="flex gap-3">
                <button type="button" onclick="closeModal('resolveModal')" 
                    class="flex-1 px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 px-4 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600">
                    <i class="fas fa-shield-alt mr-1"></i> Confirm Safe
                </button>
            </div>
        </form>
    </div>
</div>

<!-- False Alarm Modal -->
<div id="falseAlarmModal" class="fixed inset-0 bg-black/50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl w-full max-w-md mx-4">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">Mark as False Alarm - <span id="falseAlarmUserName"></span></h3>
            <button onclick="closeModal('falseAlarmModal')" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form method="POST" class="p-4">
            <input type="hidden" name="action" value="false_alarm">
            <input type="hidden" name="sos_id" id="falseAlarmSosId">
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Reason (optional)</label>
                <textarea name="notes" rows="3" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary"
                    placeholder="Why is this a false alarm?"></textarea>
            </div>
            
            <div class="flex gap-3">
                <button type="button" onclick="closeModal('falseAlarmModal')" 
                    class="flex-1 px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                    <i class="fas fa-times mr-1"></i> Mark as False
                </button>
            </div>
        </form>
    </div>
</div>

<script>
    function openResolveModal(id, name) {
        document.getElementById('resolveSosId').value = id;
        document.getElementById('resolveUserName').textContent = name;
        document.getElementById('resolveModal').classList.remove('hidden');
    }
    
    function openFalseAlarmModal(id, name) {
        document.getElementById('falseAlarmSosId').value = id;
        document.getElementById('falseAlarmUserName').textContent = name;
        document.getElementById('falseAlarmModal').classList.remove('hidden');
    }
    
    function openModal(id) {
        document.getElementById(id).classList.remove('hidden');
    }
    
    function closeModal(id) {
        document.getElementById(id).classList.add('hidden');
    }
    
    // Close modal on outside click
    document.querySelectorAll('.fixed').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });
    });
</script>

<?php
function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
